/*******************************************************************************
* Copyright (C) 2019 China Micro Semiconductor Limited Company. All Rights Reserved.
*
* This software is owned and published by:
* CMS LLC, No 2609-10, Taurus Plaza, TaoyuanRoad, NanshanDistrict, Shenzhen, China.
*
* BY DOWNLOADING, INSTALLING OR USING THIS SOFTWARE, YOU AGREE TO BE BOUND
* BY ALL THE TERMS AND CONDITIONS OF THIS AGREEMENT.
*
* This software contains source code for use with CMS
* components. This software is licensed by CMS to be adapted only
* for use in systems utilizing CMS components. CMS shall not be
* responsible for misuse or illegal use of this software for devices not
* supported herein. CMS is providing this software "AS IS" and will
* not be responsible for issues arising from incorrect user implementation
* of the software.
*
* This software may be replicated in part or whole for the licensed use,
* with the restriction that this Disclaimer and Copyright notice must be
* included with each copy of this software, whether used in part or whole,
* at all times.
*/

/****************************************************************************/
/** \file opa.c
**
**  
**
**	History:
**	
*****************************************************************************/
/****************************************************************************/
/*	include files
*****************************************************************************/
#include "opa.h"

/****************************************************************************/
/*	Local pre-processor symbols/macros('#define')
****************************************************************************/

/****************************************************************************/
/*	Global variable definitions(declared in header file with 'extern')
****************************************************************************/

/****************************************************************************/
/*	Local type definitions('typedef')
****************************************************************************/

/****************************************************************************/
/*	Local variable  definitions('static')
****************************************************************************/

/****************************************************************************/
/*	Local function prototypes('static')
****************************************************************************/

/****************************************************************************/
/*	Function implementation - global ('extern') and local('static')
****************************************************************************/

/*****************************************************************************
 ** \brief	 OPA_SetRunMode
 **			 设置运放运行模式
 ** \param [in] OPAn : OPA0、OPA1、OPA2、OPA3
 **				OpaMode:  (1)OPA_MODE_OPA
 **						  (2)OPA_MODE_ACMP
 ** \return  none
 ** \note   
 *****************************************************************************/
void OPA_SetRunMode(uint8_t OPAn, uint8_t OpaMode)
{
	uint8_t Temp=0;
	
	if(OPA0 == OPAn)
	{
		Temp = OP0CON0;
		Temp &= ~(OPA_OP0CON0_OP0FIL_Msk);
		Temp |= (OpaMode << OPA_OP0CON0_OP0FIL_Pos);
		OP0CON0 = Temp;
	}
	if(OPA1 == OPAn)
	{
		Temp = OP1CON0;
		Temp &= ~(OPA_OP1CON0_OP1FIL_Msk);
		Temp |= (OpaMode << OPA_OP1CON0_OP1FIL_Pos);	
		OP1CON0 = Temp;	
	}
	if(OPA2 == OPAn)
	{
		Temp = OP2CON0;
		Temp &= ~(OPA_OP2CON0_OP2FIL_Msk);
		Temp |= (OpaMode << OPA_OP2CON0_OP2FIL_Pos);	
		OP2CON0 = Temp;	
	}
		if(OPA1 == OPAn)
	{
		Temp = OP3CON0;
		Temp &= ~(OPA_OP3CON0_OP3FIL_Msk);
		Temp |= (OpaMode << OPA_OP3CON0_OP3FIL_Pos);	
		OP3CON0 = Temp;	
	}
}

/*****************************************************************************
 ** \brief	 OPA_ConfigPositive
 **			 设置运放正端输入
 ** \param [in] OPAn : OPA0、OPA1、OPA2、OPA3
 **				PositiveSource:  (0)OPA_POSSEL_P	:OPnP
 **								 (1)OPA_POSSEL_BG
									 (2)OPA_POSSEL_DAC0：OP0~2
									 (2)OPA_POSSEL_DAC1: OP3
 ** \return  none
 ** \note    
 *****************************************************************************/
void OPA_ConfigPositive(uint8_t OPAn, uint8_t PositiveSource)
{
	uint8_t Temp=0;
	
	if(OPA0 == OPAn)
	{
		Temp = OP0CON0;
		Temp &= ~(OPA_OP0CON0_OP0PS_Msk);
		Temp |= (PositiveSource << OPA_OP0CON0_OP0PS_Pos);
		OP0CON0 = Temp;
	}
	if(OPA1 == OPAn)
	{
		Temp = OP1CON0;
		Temp &= ~(OPA_OP1CON0_OP1PS_Msk);
		Temp |= (PositiveSource << OPA_OP1CON0_OP1PS_Pos);	
		OP1CON0 = Temp;
	}
	if(OPA2 == OPAn)
	{
		Temp = OP2CON0;
		Temp &= ~(OPA_OP2CON0_OP2PS_Msk);
		Temp |= (PositiveSource << OPA_OP2CON0_OP2PS_Pos);	
		OP2CON0 = Temp;
	}
	if(OPA3 == OPAn)
	{
		Temp = OP3CON0;
		Temp &= ~(OPA_OP3CON0_OP3PS_Msk);
		Temp |= (PositiveSource << OPA_OP3CON0_OP3PS_Pos);	
		OP3CON0 = Temp;
	}
}

/*****************************************************************************
 ** \brief	 OPA_ConfigNegative
 **			 设置运放器负端输入
 ** \param [in] OPAn : OPA0、OPA1、OPA2、OPA3
 **				NegativeSource:  (1)OPA_NEGSEL_N		:OPnN			
 ** \return  none
 ** \note   
 *****************************************************************************/
void OPA_ConfigNegative(uint8_t OPAn, uint8_t NegativeSource)
{
	uint8_t Temp=0;
	
	if(OPA0 == OPAn)
	{
		Temp = OP0CON0;
		Temp &= ~(OPA_OP0CON0_OP0NS_Msk);
		Temp |= (NegativeSource << OPA_OP0CON0_OP0NS_Pos);
		OP0CON0 = Temp;
	}
	if(OPA1 == OPAn)
	{
		Temp = OP1CON0;
		Temp &= ~(OPA_OP1CON0_OP1NS_Msk);
		Temp |= (NegativeSource << OPA_OP1CON0_OP1NS_Pos);
		OP1CON0 = Temp;
	}
	if(OPA2 == OPAn)
	{
		Temp = OP2CON0;
		Temp &= ~(OPA_OP2CON0_OP2NS_Msk);
		Temp |= (NegativeSource << OPA_OP2CON0_OP2NS_Pos);
		OP2CON0 = Temp;
	}
	if(OPA3 == OPAn)
	{
		Temp = OP3CON0;
		Temp &= ~(OPA_OP3CON0_OP3NS_Msk);
		Temp |= (NegativeSource << OPA_OP3CON0_OP3NS_Pos);
		OP3CON0 = Temp;
	}
}

/*****************************************************************************
 ** \brief	 OPA_EnableOutput
 **			 开启运放器输出端
 ** \param [in] OPAn : OPA0、OPA1、OPA2、OPA3
 ** \return  none
 ** \note   
 *****************************************************************************/
void OPA_EnableOutput(uint8_t OPAn)
{
	if(OPA0 == OPAn)
	{
		OP0CON0 |= (OPA_OP0CON0_OP0OS_Msk);
	}
	if(OPA1 == OPAn)
	{
		OP1CON0 |= (OPA_OP1CON0_OP1OS_Msk);	
	}
	if(OPA2 == OPAn)
	{
		OP2CON0 |= (OPA_OP2CON0_OP2OS_Msk);	
	}
	if(OPA3 == OPAn)
	{
		OP3CON0 |= (OPA_OP3CON0_OP3OS_Msk);	
	}
}

/*****************************************************************************
 ** \brief	 OPA_DisableOutput
 **			 关闭运放器输出端
 ** \param [in] OPAn : OPA0、OPA1、OPA2、OPA3
 ** \return  none
 ** \note   
 *****************************************************************************/
void OPA_DisableOutput(uint8_t OPAn)
{
	if(OPA0 == OPAn)
	{
		OP0CON0 &= ~(OPA_OP0CON0_OP0OS_Msk);
	}
	if(OPA1 == OPAn)
	{
		OP1CON0 &= ~(OPA_OP1CON0_OP1OS_Msk);
	}
	if(OPA2 == OPAn)
	{
		OP2CON0 &= ~(OPA_OP2CON0_OP2OS_Msk);
	}
	if(OPA3 == OPAn)
	{
		OP3CON0 &= ~(OPA_OP3CON0_OP3OS_Msk);
	}
}



/********************************************************************************
 ** \brief	 OPA_Start
 **			 开启OP
 ** \param [in] OPAn : OPA0、OPA1、OPA2、OPA3
 ** \return  none
 ** \note   
 ******************************************************************************/
void OPA_Start(uint8_t OPAn)
{
	if(OPA0 == OPAn)
	{
		OP0CON0 |= OPA_OP0CON0_OP0EN_Msk;
	}
	if(OPA1 == OPAn)
	{
		OP1CON0 |= OPA_OP1CON0_OP1EN_Msk;	
	}
	if(OPA2 == OPAn)
	{
		OP2CON0 |= OPA_OP2CON0_OP2EN_Msk;	
	}
	if(OPA3 == OPAn)
	{
		OP3CON0 |= OPA_OP3CON0_OP3EN_Msk;	
	}
}

/********************************************************************************
 ** \brief	 OPA_Stop
 **			 关闭OPA
 ** \param [in] OPAn : OPA0、OPA1、OPA2、OPA3
 ** \return  none
 ** \note   
 ******************************************************************************/
void OPA_Stop(uint8_t OPAn)
{
	if(OPA0 == OPAn)
	{
		OP0CON0 &= ~OPA_OP0CON0_OP0EN_Msk;
	}
	if(OPA1 == OPAn)
	{
		OP1CON0 &= ~OPA_OP1CON0_OP1EN_Msk;	
	}	
	if(OPA2 == OPAn)
	{
		OP2CON0 &= ~OPA_OP2CON0_OP2EN_Msk;	
	}
	if(OPA3 == OPAn)
	{
		OP3CON0 &= ~OPA_OP3CON0_OP3EN_Msk;	
	}
}

/********************************************************************************
 ** \brief	 OPA_GetResult
 **			 获取结果
 ** \param [in] OPAn : OPA0、OPA1、OPA2、OPA3
 ** \return  0/1
 ** \note   
 ******************************************************************************/
uint8_t OPA_GetResult(uint8_t OPAn)
{
	if(OPA0 == OPAn)
	{
		return ((OP0CON1 & OPA_OP0CON1_OP0DOUT_Msk)? 1:0);
	}
	if(OPA1 == OPAn)
	{
		return ((OP1CON1 & OPA_OP1CON1_OP1DOUT_Msk)? 1:0);
	}	
	if(OPA2 == OPAn)
	{
		return ((OP2CON1 & OPA_OP2CON1_OP2DOUT_Msk)? 1:0);
	}	
	if(OPA3 == OPAn)
	{
		return ((OP3CON1 & OPA_OP3CON1_OP3DOUT_Msk)? 1:0);
	}	
	return 0;
}


/*****************************************************************************
 ** \brief	 OPA_ConfigOffsetAdj
 **			 配置OPA失调电压调节方式
 ** \param [in] OPAn : OPA0、OPA1、OPA2、OPA3
 **				OffsetAdj:	(1)OPA_OFFSET_CONFIG
 **							(2)OPA_OFFSET_OPADJ
 **				AdjVlue : 失调电压调节值：0x00~0x1f
 ** \return  none
 ** \note   
*****************************************************************************/
void OPA_ConfigOffsetAdj(uint8_t OPAn,uint8_t OffsetAdj, uint8_t AdjVlue)
{
	uint8_t Temp=0;
	
	if(OPA0 == OPAn)
	{
		Temp = OP0CON1;
		Temp &= ~(OPA_OP0CON1_OP0ADJ_Msk);
		Temp |= 0x1f & AdjVlue;
		OP0CON1 = Temp;
		OP0ADJE = OffsetAdj;
	}
	if(OPA1 == OPAn)
	{
		Temp = OP1CON1;
		Temp &= ~(OPA_OP1CON1_OP1ADJ_Msk);
		Temp |= 0x1f & AdjVlue;
		OP1CON1 = Temp;
		OP1ADJE = OffsetAdj;
	}	
	if(OPA2 == OPAn)
	{
		Temp = OP2CON1;
		Temp &= ~(OPA_OP2CON1_OP2ADJ_Msk);
		Temp |= 0x1f & AdjVlue;
		OP2CON1 = Temp;
		OP2ADJE = OffsetAdj;
	}
	if(OPA3 == OPAn)
	{
		Temp = OP3CON1;
		Temp &= ~(OPA_OP3CON1_OP3ADJ_Msk);
		Temp |= 0x1f & AdjVlue;
		OP3CON1 = Temp;
		OP3ADJE = OffsetAdj;
	}	
}
/*****************************************************************************
 ** \brief	 OPA_Delay
 **			 延时函数
 ** \param [in]none
 ** \return  none
 ** \note   
*****************************************************************************/
static void OPA_Delay(void)
{
	volatile uint8_t i;
	for(i=0;i<50;i++)
		_nop_();
}
/*****************************************************************************
 ** \brief	 OPA_GetOffsetAdjValue
 **			 获取OPA失调电压调节值
 ** \param [in] OPAn : OPA0、OPA1、OPA2、OPA3
 **				OpaMode:  (1)OPA_ADJ_INGPUT_SEL_NEG    调节模式输入端选择负端
 **						  (2)OPA_ADJ_INGPUT_SEL_POS	   调节模式输入端选择正端
 ** \return  5位失调电压修调值
 ** \note   
 *****************************************************************************/
uint8_t  OPA_GetOffsetAdjValue(uint8_t OPAn, uint8_t OpaMode)
{
	uint8_t temp, OPAOut, Adjvalue;
	Adjvalue =0;	
	if(OPA0 == OPAn)
	{
		/*(1)关闭OPA0*/		
		OPA_Stop(OPA0);
		/*(2)设置修调模式下的信号输入端*/
		
//		temp = OP0CON1;
		temp = 0x00;	
		temp |= (OpaMode<< OPA_OP0CON1_OP0CRS_Pos);
		OP0CON1 = temp;

		OP0CON0 &= 0xF0; 	
		GPIO_SET_MUX_MODE( P31CFG , GPIO_P31_MUX_OP0N);
		GPIO_SET_MUX_MODE( P32CFG , GPIO_P32_MUX_OP0P);	
				
		/*（3）开启OPA0调节模式*/
		OP0CON0 |= (1<<6);	
		/*(4)设置OPA的调节值源为OPA0ADJ<4:0>*/
		OP0ADJE = 0xAA;
		/*(5)开启OPA模块*/
		OPA_Start(OPA0);
		/*(6)获取调节值*/
		OPA_Delay();
		OPAOut = OPA_GetResult(OPA0);
		for(temp=0; temp<0x20; temp++)
		{
			OP0CON1 = temp;
			OPA_Delay();
			if(OPAOut != OPA_GetResult(OPA0))
			{
				Adjvalue = temp;
				break;
			}
		}
		/*(7)关闭OPA修调模式*/
		OP0CON0 &= ~(1<<6);
		OPA_Stop(OPA0);
		return 	Adjvalue;
	}	

	if(OPA1 == OPAn)
	{
		/*(1)关闭OPA1*/		
		OPA_Stop(OPA1);
		OPA_SetRunMode(OPA1,OpaMode);
		/*(2)设置修调模式下的信号输入端*/
//		temp = OP1CON1;
		temp = 0x00;	
		temp |= (OpaMode<< OPA_OP0CON1_OP0CRS_Pos);
		OP1CON1 = temp;

		OP1CON0 &= 0xF0; 	
		GPIO_SET_MUX_MODE( P22CFG , GPIO_P22_MUX_OP1P);
		GPIO_SET_MUX_MODE( P23CFG , GPIO_P23_MUX_OP1N);		
			
		/*（3）开启OPA1调节模式*/
		OP1CON0 |= (1<<6);	
		/*(4)设置OPA的调节值源为OPA1ADJ<4:0>*/
		OP1ADJE = 0xAA;
		/*(5)开启OPA模块*/
		OPA_Start(OPA1);
		/*(6)获取调节值*/
		OPA_Delay();
		OPAOut = OPA_GetResult(OPA1);
		for(temp=0; temp<0x20; temp++)
		{
			OP1CON1 = temp;
			OPA_Delay();
			if(OPAOut != OPA_GetResult(OPA1))
			{
				Adjvalue = temp;
				break;
			}
		}
		/*(7)关闭OPA修调模式*/
		OP1CON0 &= ~(1<<6);
		OPA_Stop(OPA1);
		
		return 	Adjvalue;
	}
	if(OPA2 == OPAn)
	{
		/*(1)关闭OPA2*/		
		OPA_Stop(OPA2);
		/*(2)设置修调模式下的信号输入端*/
//		temp = OP2CON1;
		temp = 0x00;	
		temp |= (OpaMode<< OPA_OP2CON1_OP2CRS_Pos);
		OP2CON1 = temp;

		OP2CON0 &= 0xF0; 	
		GPIO_SET_MUX_MODE( P14CFG , GPIO_P14_MUX_OP2N);
		GPIO_SET_MUX_MODE( P15CFG , GPIO_P15_MUX_OP2P);	
				
		/*（3）开启OPA2调节模式*/
		OP2CON0 |= (1<<6);	
		/*(4)设置OPA的调节值源为OPA2ADJ<4:0>*/
		OP2ADJE = 0xAA;
		/*(5)开启OPA模块*/
		OPA_Start(OPA2);
		/*(6)获取调节值*/
		OPA_Delay();
		OPAOut = OPA_GetResult(OPA2);
		for(temp=0; temp<0x20; temp++)
		{
			OP2CON1 = temp;
			OPA_Delay();
			if(OPAOut != OPA_GetResult(OPA2))
			{
				Adjvalue = temp;
				break;
			}
		}
		/*(7)关闭OPA修调模式*/
		OP2CON0 &= ~(1<<6);
		OPA_Stop(OPA2);
		return 	Adjvalue;
	}	
	if(OPA3 == OPAn)
	{
		/*(1)关闭OPA3*/		
		OPA_Stop(OPA3);
		/*(2)设置修调模式下的信号输入端*/
//		temp = OP3CON1;
		temp = 0x00;	
		temp |= (OpaMode<< OPA_OP3CON1_OP3CRS_Pos);
		OP3CON1 = temp;

		OP3CON0 &= 0xF0; 	
		GPIO_SET_MUX_MODE( P17CFG , GPIO_P17_MUX_OP3N);
				
		/*（3）开启OPA3调节模式*/
		OP3CON0 |= (1<<6);	
		/*(4)设置OPA的调节值源为OPA3ADJ<4:0>*/
		OP3ADJE = 0xAA;
		/*(5)开启OPA模块*/
		OPA_Start(OPA3);
		/*(6)获取调节值*/
		OPA_Delay();
		OPAOut = OPA_GetResult(OPA3);
		for(temp=0; temp<0x20; temp++)
		{
			OP3CON1 = temp;
			OPA_Delay();
			if(OPAOut != OPA_GetResult(OPA3))
			{
				Adjvalue = temp;
				break;
			}
		}
		/*(7)关闭OPA修调模式*/
		OP3CON0 &= ~(1<<6);
		OPA_Stop(OPA3);
		return 	Adjvalue;
	}	
	return 0;	
}
