/*******************************************************************************
* Copyright (C) 2019 China Micro Semiconductor Limited Company. All Rights Reserved.
*
* This software is owned and published by:
* CMS LLC, 21F, Building T1, Qianhai Financial Center, No.91 Guiwan 3rd Road, 
* Qianhai Shenzhen-Hong Kong Cooperation Zone, Shenzhen City , Guangdong Province , 
* P R China.
*
* BY DOWNLOADING, INSTALLING OR USING THIS SOFTWARE, YOU AGREE TO BE BOUND
* BY ALL THE TERMS AND CONDITIONS OF THIS AGREEMENT.
*
* This software contains source code for use with CMS
* components. This software is licensed by CMS to be adapted only
* for use in systems utilizing CMS components. CMS shall not be
* responsible for misuse or illegal use of this software for devices not
* supported herein. CMS is providing this software "AS IS" and will
* not be responsible for issues arising from incorrect user implementation
* of the software.
*
* This software may be replicated in part or whole for the licensed use,
* with the restriction that this Disclaimer and Copyright notice must be
* included with each copy of this software, whether used in part or whole,
* at all times.
*/

/****************************************************************************/
/** \file timer.c
**
**  
**
**	History:
**	
****************************************************************************/
/***************************************************************************/
/*	include files
****************************************************************************/
#include "timer.h"

/***************************************************************************/
/*	Local pre-processor symbols/macros('#define')
****************************************************************************/

/***************************************************************************/
/*	Global variable definitions(declared in header file with 'extern')
****************************************************************************/

/***************************************************************************/
/*	Local type definitions('typedef')
****************************************************************************/

/***************************************************************************/
/*	Local variable  definitions('static')
****************************************************************************/

/***************************************************************************/
/*	Local function prototypes('static')
****************************************************************************/

/***************************************************************************/
/*	Function implementation - global ('extern') and local('static')
****************************************************************************/
/****************************************************************************
 ** \brief	TMR_ConfigRunMode
 **			配置定时器运行模式
 ** \param [in] Timern	：		TMR0,TMR1
 **				TimerMode：(1)TMR_MODE_TIMING	:定时模式
 **							     (2)TMR_MODE_COUNT	:计数模式
 **				TimerModeBranch	：模式的分支
 **								(1)TMR_TIM_13BIT 	:13位定时/计数模式
 **								(2)TMR_TIM_16BIT 	:16位定时/计数模式		
 **								(3)TMR_TIM_AUTO_8BIT :8位自动重载定时/计数模式
 **								(4)TMR_TIM_TWO_INDEPENDENT_8BIT	:两个独立的8位定时/计数模式
 ** \return  none
 ** \note    此函数不支持TMR2
 *****************************************************************************/
void TMR_ConfigRunMode(uint8_t Timern, uint8_t TimerMode, uint8_t TimerModeBranch)
{
	switch (Timern)
	{
		case TMR0:
			TMOD &= ~(TMR_TMOD_T0Mn_Msk | TMR_TMOD_CT0_Msk);
			TMOD |= (TimerMode<< TMR_TMOD_CT0_Pos) | TimerModeBranch;
			break;
		case TMR1:
			TMOD &= ~(TMR_TMOD_T1Mn_Msk | TMR_TMOD_CT1_Msk);
			TMOD |= (TimerMode<< TMR_TMOD_CT1_Pos) | (TimerModeBranch<< TMR_TMOD_T1Mn_Pos);
			break;		
		default:
			break;
	}		
}
/***************************************************************************
 ** \brief	TMR_ConfigTimerClk
 **			配置定时器运行时钟
 ** \param [in] Timern：TMR0,TMR1
 **				TimerClkDiv	：(1)TMR_CLK_DIV_4 
 **							        (2)TMR_CLK_DIV_12
 ** \return  none
 ** \note   
*****************************************************************************/
void TMR_ConfigTimerClk(uint8_t Timern,uint8_t TimerClkDiv)
{
	switch (Timern)
	{
		case TMR0:   
			CKCON &= ~(SYS_CKCON_T0M_Msk);
			CKCON |= (TimerClkDiv << SYS_CKCON_T0M_Pos);
			break;
		case TMR1:
			CKCON &= ~(SYS_CKCON_T1M_Msk);
			CKCON |= (TimerClkDiv << SYS_CKCON_T1M_Pos);
			break;		
		default:
			break;
	}	
}
/***************************************************************************
 ** \brief	TMR_ConfigTimerPeriod
 **			配置定时器定时周期
 ** \param [in] Timern：TMR0,TMR1
 **				TimerPeriodH:(1)TH0 (2)TH1 
 **				TimerPeriodL:(1)TL0 (2)TL1 
 ** \return  none
 ** \note    13位定时器 ：THx[7:0],TLx[4:0]
*****************************************************************************/
void TMR_ConfigTimerPeriod(uint8_t Timern , uint8_t TimerPeriodHigh, uint8_t TimerPeriodLow)
{
	switch (Timern)
	{
		case TMR0:
			TH0 = TimerPeriodHigh;
			TL0 = TimerPeriodLow;
			break;
		case TMR1:
			TH1 = TimerPeriodHigh;
			TL1 = TimerPeriodLow;
			break;		
		default:
			break;
	}		
}

/*****************************************************************************
 ** \brief	TMR_EnableGATE
 **			使能定时器门控功能
 ** \param [in] Timern	：	TMR0,TMR1
 ** \return  none
 ** \note   
 *****************************************************************************/
void TMR_EnableGATE(uint8_t Timern)
{
	if(TMR0 == Timern)
	{
		TMOD |= TMR_TMOD_GATE0_Msk;
	}
	if(TMR1 == Timern)
	{
		TMOD |= TMR_TMOD_GATE1_Msk;
	}
}
/*****************************************************************************
 ** \brief	TMR_DisableGATE
 **			关闭定时器门控
 ** \param [in] Timern	：	TMR0,TMR1
 ** \return  none
 ** \note   
 *****************************************************************************/
void TMR_DisableGATE(uint8_t Timern)
{
	if(TMR0 == Timern)
	{
		TMOD &= ~(TMR_TMOD_GATE0_Msk);
	}
	if(TMR1 == Timern)
	{
		TMOD &= ~(TMR_TMOD_GATE1_Msk);
	}
}
/*****************************************************************************
 ** \brief	TMR_GetCountValue
 **			获取定时器计数模式的计数值
 ** \param [in] Timern	：	TMR0,TMR1
 ** \return  16bits的计数值：[THn:TLn]
 ** \note   模式2/3: 请酌情使用
 *****************************************************************************/
uint16_t  TMR_GetCountValue(uint8_t Timern)
{
	if(	TMR0 == Timern)
	{	
		if( TMR_TIM_13BIT == (TMOD & TMR_TMOD_T0Mn_Msk))
		{
			return(((TH0<< 5) | TL0));
		}
		return(((TH0<< 8) | TL0));		
	}	
	if(	TMR1 == Timern)
	{
		if( TMR_TIM_13BIT == (TMOD & TMR_TMOD_T1Mn_Msk))
		{
			return(((TH1<< 5) | TL1));		
		}
		return(((TH1<< 8) | TL1));			
	}
	return 0;
}

/*****************************************************************************
 ** \brief	TMR_EnableOverflowInt
 **			使能定时器溢出中断
 ** \param [in] Timern	：	TMR0,TMR1
 ** \return   none
 ** \note   
 *****************************************************************************/
void  TMR_EnableOverflowInt(uint8_t Timern)
{
	switch (Timern)
	{
		case TMR0:
			ET0 = 1;
			break;
		case TMR1:
			ET1 = 1;
			break;	
		default:
			break;
	}		
}

/*****************************************************************************
 ** \brief	TMR_DisableOverflowInt
 **			关闭定时器溢出中断
 ** \param [in] Timern	：	TMR0,TMR1
 ** \return   none
 ** \note   
 *****************************************************************************/
void  TMR_DisableOverflowInt(uint8_t Timern)
{
	switch (Timern)
	{
		case TMR0:
			ET0 = 0;
			break;
		case TMR1:
			ET1 = 0;
			break;		
		default:
			break;
	}	
}
/*****************************************************************************
 ** \brief	TMR_GetOverflowIntFlag
 **			获取定时器中断标志
 ** \param [in] Timern	：	TMR0,TMR1
 ** \return   0：无中断， 1：有中断
 ** \note   定时器0、1使用模式3时，其中TLn的溢出中断使用TMR0中断标志
 **									   THn的溢出中断使用TMR1中断标志 
 *****************************************************************************/
uint8_t TMR_GetOverflowIntFlag(uint8_t Timern)
{
	uint8_t IntFlag =0;
	switch (Timern)
	{
		case TMR0:
			IntFlag = TCON & TMR_TCON_TF0_Msk;
			break;
		case TMR1:
			IntFlag = TCON & TMR_TCON_TF1_Msk;
			break;	
		default:
			break;
	}
	return (IntFlag ? 1:0);
}
/*****************************************************************************
 ** \brief	TMR_ClearOverflowIntFlag
 **			清除定时器溢出中断标志
 ** \param [in] Timern	：	TMR0,TMR1
 ** \return  none
 ** \note   
 *****************************************************************************/
void  TMR_ClearOverflowIntFlag(uint8_t Timern)
{
	switch (Timern)
	{
		case TMR0:
			TF0 = 0;
			break;
		case TMR1:
			TF1 = 0;
			break;		
		default:
			break;
	}	
}
/*****************************************************************************
 ** \brief	TMR_Start
 **			开启定时器
 ** \param [in] Timern	：	TMR0,TMR1
 ** \return  none
 ** \note   
 *****************************************************************************/
void TMR_Start(uint8_t Timern)
{
	switch (Timern)
	{
		case TMR0:
			TR0 = 1;
			break;
		case TMR1:
			TR1 = 1;
			break;	
		default:
			break;
	}		
}
/*****************************************************************************
 ** \brief	TMR_Stop
 **			关闭定时器
 ** \param [in] Timern	：	TMR0,TMR1
 ** \return  none
 ** \note   
 *****************************************************************************/
void TMR_Stop(uint8_t Timern)
{
	switch (Timern)
	{
		case TMR0:
			TR0 =0;
			break;
		case TMR1:
			TR1 =0;
			break;		
		default:
			break;
	}		
}


/****************************************************************************/
/*****************************************************************************
 ** \brief	TMR2_ConfigRunMode
 **			配置Timer2运行模式
 ** \param [in]	Timer2Mode 	:	(1)TMR2_MODE_TIMING	:定时模式
 **								(2)TMR2_MODE_COUNT	:计数模式
 **			   Timer2LoadMode	：	
 **								(1)TMR2_AUTO_LOAD	:定时器2自动重载模式
 **								(2)TMR2_T2EX_LOAD	:定时器2外部触发重载模式
**								(3)TMR2_LOAD_DISBALE: 禁止重载
 ** \return  none
 ** \note    
*****************************************************************************/
void TMR2_ConfigRunMode(  uint8_t Timer2Mode, uint8_t Timer2LoadMode)
{
	T2CON &= ~(TMR_T2CON_T2Rn_Msk | TMR_T2CON_T2In_Msk);
	T2CON |= Timer2Mode | Timer2LoadMode ;		
}
/*****************************************************************************
 ** \brief	TMR2_ConfigClkSource
 **			配置定时器运行时钟
 ** \param [in]	TimerClkSr	：TMR2_CLKSr_Fsys 、TMR2_CLKSr_T2PS
 ** \return  none
 ** \note   
 *****************************************************************************/
void TMR2_ConfigClkSource(uint8_t TimerClkSr)
{
	CKCON &= ~(SYS_CKCON_T2M_Msk);
	CKCON |= (TimerClkSr);
}
/*****************************************************************************
 ** \brief	TMR2_ConfigTimerClk
 **			配置定时器运行时钟
 ** \param [in]	TimerClkDiv	：TMR2_CLK_DIV_12 、TMR2_CLK_DIV_24
 ** \return  none
 ** \note   
 *****************************************************************************/
void TMR2_ConfigTimerClk(uint8_t TimerClkDiv)
{
	T2CON &= ~(TMR_T2CON_T2PS_Msk);
	T2CON |= (TimerClkDiv);
}
/***************************************************************************
 ** \brief	TMR2_ConfigTimerPeriod
 **			配置定时器2定时周期
 ** \param [in] TimerPeriod；定时器2定时周期
 ** \return  none
 ** \note   
*****************************************************************************/
void TMR2_ConfigTimerPeriod(uint16_t TimerPeriod)
{
	TL2 = (uint8_t)TimerPeriod;
	TH2 = (uint8_t)(TimerPeriod >>8);
	if((TMR2_AUTO_LOAD == (T2CON & TMR_T2CON_T2Rn_Msk)) || (TMR2_T2EX_LOAD == (T2CON & TMR_T2CON_T2Rn_Msk)))
	{
		RLDL = (uint8_t)TimerPeriod;			//自动重载或T2EX下降沿加载
		RLDH = (uint8_t)(TimerPeriod >>8);		
	}
}
/*****************************************************************************
 ** \brief	TMR2_EnableGATE
 **			使能定时器2门控功能
 ** \param [in] none
 ** \return  none
 ** \note   
 *****************************************************************************/
void TMR2_EnableGATE(void)
{
	T2CON |= TMR_T2CON_T2In_Msk;
}
/*****************************************************************************
 ** \brief	TMR2_DisableGATE
 **			关闭定时器2门控功能
 ** \param [in] none
 ** \return  none
 ** \note   
 *****************************************************************************/
void TMR2_DisableGATE(void)
{
	uint8_t Temp=0;
	Temp = T2CON;
	Temp &= ~(TMR_T2CON_T2In_Msk);	
	Temp |= (0x1 << TMR_T2CON_T2In_Pos);
	T2CON = Temp;
}

/*****************************************************************************
 ** \brief	TMR2_EnableCompare
 **			使能比较模式
 ** \param [in]	Timer2CCn:  TMR2_CC0 ~ TMR2_CC1
 **				CompareMode :(1)TMR2_CMP_MODE_0
 **							 (2)TMR2_CMP_MODE_1
 ** \return  none
 ** \note   
 *****************************************************************************/
void TMR2_EnableCompare(uint8_t Timer2CCn, uint8_t CompareMode)
{
	if(TMR2_CMP_MODE_0 == CompareMode)
	{
		T2CM = 0;
	}
	if(TMR2_CMP_MODE_1 == CompareMode)
	{
		T2CM = 1;
	}	
	if( Timer2CCn == TMR2_CC0)
	{
		CCEN &= ~(TMR_CCEN_CMn0_Msk);
		CCEN |= (TMR2_MODE_COMPARE << TMR_CCEN_CMn0_Pos);
	}
	if( Timer2CCn == TMR2_CC1)
	{
		CCEN &= ~(TMR_CCEN_CMn1_Msk);
		CCEN |= (TMR2_MODE_COMPARE << TMR_CCEN_CMn1_Pos);
	}
	
}
/*****************************************************************************
 ** \brief	TMR2_DisableCompare
 **			关闭比较模式
 ** \param [in]	Timer2CCMask:  TMR2_CC0 ~ TMR2_CC1
 ** \return  none
 ** \note   
 *****************************************************************************/
void TMR2_DisableCompare(uint8_t Timer2CCn)
{
	if( Timer2CCn == TMR2_CC0)
	{           
		CCEN &= ~(TMR_CCEN_CMn0_Msk);
	}
	if( Timer2CCn == TMR2_CC1)
	{           
		CCEN &= ~(TMR_CCEN_CMn1_Msk);
	}		
}

/*****************************************************************************
 ** \brief	TMR2_ConfigCompareValue
 **			配置比较通道比较值
 ** \param [in]	Timer2CCMask:  TMR2_CC0 ~ TMR2_CC1
 **				CompareValue:  0x0 ~ 0xFFFF;
 ** \return  none
 ** \note   
 *****************************************************************************/
void TMR2_ConfigCompareValue(uint8_t Timer2CCn, uint16_t CompareValue)
{
	if( TMR2_CC0 == Timer2CCn)
	{
		RLDL = (uint8_t)CompareValue;
		RLDH = (uint8_t)(CompareValue >>8);
	}
	if( TMR2_CC1 == Timer2CCn)
	{
		CCL1 = (uint8_t)CompareValue;
		CCH1 = (uint8_t)(CompareValue >>8);	
	}
}
/*****************************************************************************
 ** \brief	TMR2_ConfigCompareIntMode
 **			配置比较通道比较中断模式
 ** \param [in]	Timer2CompareIntMode:	(1)TMR2_CMP_INT_MODE0   
 **										(2)TMR2_CMP_INT_MODE1	  
 ** \return  none
 ** \note   
 **						 
 *****************************************************************************/
void TMR2_ConfigCompareIntMode(uint8_t Timer2CompareIntMode)
{
	T2CON &= ~(TMR_T2CON_I3FR_Msk);
	T2CON |= Timer2CompareIntMode;
}

/*****************************************************************************/
/*****************************************************************************/
/*****************************************************************************
 ** \brief	TMR2_EnableCapture
 **			使能捕获模式
 ** \param [in]	Timer2CCMask:  TMR2_Capture0 ~ TMR2_Capture1
 **				Timer2CaptureMode: (1)TMR2_CAP_WRITE_REGISTER   捕获通道在写寄存器时捕获
 **								   (2)TMR2_CAP_EDGE_FALLING		捕获通道下降沿捕获			  	
 **								   (3)TMR2_CAP_EDGE_RISING		捕获通道上升沿捕获
 **								   (4)TMR2_CAP_EDGE_BOTH		捕获通道在上升和下降边沿都可捕获							   
 ** \return  none
 ** \note  
 *****************************************************************************/
void TMR2_EnableCapture(uint8_t Timer2CCn, uint8_t Timer2CaptureMode)
{
	if( TMR2_Capture0 == Timer2CCn)
	{
		T2CON &= ~(TMR_T2CON_I3FR_Msk);
		T2CON |= ( (0x1 & Timer2CaptureMode)<<TMR_T2CON_I3FR_Pos);		
		CCEN &= ~(TMR_CCEN_CMn0_Msk);
		CCEN |= ( (0x1 | Timer2CaptureMode)<< TMR_CCEN_CMn0_Pos);
	}
	if( TMR2_Capture1 == Timer2CCn)
	{
		T2CON &= ~(TMR_T2CON_CAPES_Msk);
		if(TMR2_CAP_EDGE_FALLING == Timer2CaptureMode)
		{
			T2CON |= ( (0x1)<<TMR_T2CON_CAPES_Pos);			
		}	
		CCEN &= ~(TMR_CCEN_CMn1_Msk);
		CCEN |= ((0x1 | Timer2CaptureMode)<< TMR_CCEN_CMn1_Pos);
		
	}
}

/*****************************************************************************
 ** \brief	TMR2_DisableCapture
 **			关闭捕获模式
 ** \param [in]	Timer2CCn:  TMR2_Capture0 ~ TMR2_Capture1
 ** \return  none
 ** \note   
*****************************************************************************/
void TMR2_DisableCapture(uint8_t Timer2CCn)
{
	if( Timer2CCn == TMR2_Capture0)
	{
		CCEN &= ~(TMR_CCEN_CMn0_Msk);
	}
	if( Timer2CCn == TMR2_Capture1)
	{
		CCEN &= ~(TMR_CCEN_CMn1_Msk);
	}	
}

/*****************************************************************************
 ** \brief	TMR2_GetCaptureValue
 **			获取捕获值
 ** \param [in]	Timer2CCn:  TMR2_Capture0 ~ TMR2_Capture1			
 ** \return  16bit value
 ** \note   
 *****************************************************************************/
uint16_t TMR2_GetCaptureValue(uint8_t Timer2CCn)
{
	uint16_t  CaputerValue = 0;
	switch(Timer2CCn)
	{
		case TMR2_Capture0:
			CaputerValue = RLDH;
			CaputerValue = (CaputerValue <<8) | RLDL;
			break;
		case TMR2_Capture1:
			CaputerValue = CCH1;
			CaputerValue = (CaputerValue <<8) | CCL1;
			break;		
		default:
			break;
	}		
	return CaputerValue;	
}

/*****************************************************************************
 ** \brief	TMR2_EnableOverflowInt
 **			使能定时器2溢出中断
 ** \param [in] none
 ** \return   none
 ** \note   
 *****************************************************************************/
void TMR2_EnableOverflowInt(void)
{
	T2IE |= IRQ_T2IE_T2OVIE_Msk;
}

/*****************************************************************************
 ** \brief	TMR2_DisableOverflowInt
 **			关闭定时器2溢出中断
 ** \param [in] none
 ** \return   none
 ** \note   
 *****************************************************************************/
void TMR2_DisableOverflowInt(void)
{
	T2IE &= ~(IRQ_T2IE_T2OVIE_Msk);
}

/*****************************************************************************
 ** \brief	TMR2_GetOverflowIntFlag
 **			获取定时器2溢出中断标志
 ** \param [in] none
 ** \return   0：无中断， 1：有中断
 ** \note   
 *****************************************************************************/
uint8_t TMR2_GetOverflowIntFlag(void)
{
	return((T2IF & IRQ_T2IF_T2F_Msk)? 1:0);
}
/*****************************************************************************
 ** \brief	TMR2_ClearOverflowIntFlag
 **			清除定时器2溢出中断标志
 ** \param [in] none
 ** \return   none
 ** \note   
 *****************************************************************************/
void TMR2_ClearOverflowIntFlag(void)
{
	T2IF =  0xff & (~(IRQ_T2IF_T2F_Msk));
}

/*****************************************************************************
 ** \brief	TMR2_EnableT2EXInt
 **			使能定时器2外部加载T2EX中断
 ** \param [in] none
 ** \return   none
 ** \note   
 *****************************************************************************/
void TMR2_EnableT2EXInt(void)
{
	T2IE |= IRQ_T2IE_T2EXIE_Msk;
}

/*****************************************************************************
 ** \brief	TMR2_DisableT2EXInt
 **			关闭定时器2外部加载T2EX中断
 ** \param [in] none
 ** \return   none
 ** \note   
 *****************************************************************************/
void TMR2_DisableT2EXInt(void)
{
	T2IE &= ~(IRQ_T2IE_T2EXIE_Msk);
}
/*****************************************************************************
 ** \brief	TMR_GetT2EXIntFlag
 **			获取定时器2外部加载T2EX中断标志
 ** \param [in] none
 ** \return   0：无中断， 1：有中断
 ** \note   
 *****************************************************************************/
uint8_t TMR2_GetT2EXIntFlag(void)
{
	return((T2IF & IRQ_T2IF_T2EXIF_Msk)? 1:0);
}

/*****************************************************************************
 ** \brief	TMR2_ClearT2EXIntFlag
 **			清除定时器2外部加载T2EX中断标志
 ** \param [in] none
 ** \return   none
 ** \note   
 *****************************************************************************/
void TMR2_ClearT2EXIntFlag(void)
{
	T2IF =  0xff & (~(IRQ_T2IF_T2EXIF_Msk));
}

/*****************************************************************************
 ** \brief	TMR2_EnableCompareInt
 **			使能定时器2通道比较中断
 ** \param [in]Timer2CCn:  TMR2_CC0 ~ TMR2_CC1
 ** \return   none
 ** \note   
 *****************************************************************************/
void TMR2_EnableCompareInt(uint8_t Timer2CCn)
{
	T2IE |= (0x1<<Timer2CCn);
}

/*****************************************************************************
 ** \brief	TMR2_DisableCompareInt
 **			关闭定时器2通道比较中断
 ** \param [in] Timer2CCn:  TMR2_CC0 ~ TMR2_CC1
 ** \return   none
 ** \note   
 *****************************************************************************/
void TMR2_DisableCompareInt(uint8_t Timer2CCn)
{
	T2IE &= ~(0x1<<Timer2CCn);
}

/*****************************************************************************
 ** \brief	TMR2_GetCompareIntFlag
 **			获取定时器2通道比较中断标志
 ** \param [in]Timer2CCn:  TMR2_CC0 ~ TMR2_CC1
 ** \return   0：无中断， 1：有中断
 ** \note   
 *****************************************************************************/
uint8_t TMR2_GetCompareIntFlag(uint8_t Timer2CCn)
{
	return ((T2IF & (0x1<< Timer2CCn))? 1:0);
}
/*****************************************************************************
 ** \brief	TMR2_ClearCompareIntFlag
 **			清除定时器2通道比较中断标志
 ** \param [in]Timer2CCn:  TMR2_CC0 ~ TMR2_CC1
 ** \return   none
 ** \note   
 *****************************************************************************/
void TMR2_ClearCompareIntFlag(uint8_t Timer2CCn)
{
	T2IF = 0xff & (~(0x1<< Timer2CCn));
}

/*****************************************************************************
 ** \brief	TMR2_EnableCaptureInt
 **			使能定时器2通道捕获中断
 ** \param [in]Timer2CCn:  TMR2_Capture0 ~ TMR2_Capture1
 ** \return   none
 ** \note   
 *****************************************************************************/
void TMR2_EnableCaptureInt(uint8_t Timer2CCn)
{
	T2IE |= (0x1<<Timer2CCn);
}
/*****************************************************************************
 ** \brief	TMR2_DisableCaptureInt
 **			关闭定时器2通道捕获中断
 ** \param [in]Timer2CCn:  TMR2_Capture0 ~ TMR2_Capture1
 ** \return   none
 ** \note   
 *****************************************************************************/
void TMR2_DisableCaptureInt(uint8_t Timer2CCn)
{
	T2IE &= ~(0x1<<Timer2CCn);
}
/*****************************************************************************
 ** \brief	TMR2_GetCaptureIntFlag
 **			获取定时器2通道捕获中断标志
 ** \param [in]Timer2CCn:  TMR2_Capture0 ~ TMR2_Capture1
 ** \return   0：无中断， 1：有中断
 ** \note   
 *****************************************************************************/
uint8_t TMR2_GetCaptureIntFlag(uint8_t Timer2CCn)
{
	return ((T2IF & (0x1<< Timer2CCn))? 1:0);
}
/*****************************************************************************
 ** \brief	TMR2_ClearCaptureIntFlag
 **			清除定时器2通道捕获中断标志
 ** \param [in]Timer2CCn:  TMR2_Capture0 ~ TMR2_Capture1
 ** \return   none
 ** \note   
 *****************************************************************************/
void TMR2_ClearCaptureIntFlag(uint8_t Timer2CCn)
{
	T2IF = 0xff & (~(0x1<< Timer2CCn));
}

/*****************************************************************************
 ** \brief	TMR2_AllIntEnable
 **			使能定时器2总中断
 ** \param [in] none
 ** \return   none
 ** \note   
 *****************************************************************************/
void TMR2_AllIntEnable(void)
{
	ET2 = 1;
}

/*****************************************************************************
 ** \brief	TMR2_AllIntDisable
 **			关闭定时器2总中断
 ** \param [in] none
 ** \return   none
 ** \note   
 *****************************************************************************/
void TMR2_AllIntDisable(void)
{
	ET2 = 0;
}
/*****************************************************************************
 ** \brief	TMR2_Start
 **			开启定时器2
 ** \param [in] none
 ** \return  none
 ** \note   
 *****************************************************************************/
void TMR2_Start(void)
{
	if(0x0 == (T2CON & TMR_T2CON_T2In_Msk))
	{
		T2CON |= (0x1 <<TMR_T2CON_T2In_Pos);	
	}
}
/*****************************************************************************
 ** \brief	TMR2_Stop
 **			关闭定时器2
 ** \param [in] none
 ** \return  none
 ** \note   
 *****************************************************************************/
void TMR2_Stop(void)
{
	T2CON &= ~(TMR_T2CON_T2In_Msk);
}
/*****************************************************************************
 ** \brief	TMR5_Start
 **			使能定时器5
 ** \param [in] none
 ** \return   none
 ** \note   									  
 *****************************************************************************/
void TMR5_Start(void)
{
	T5CON |= TMR_T5CON_T5E_Msk;
}
/*****************************************************************************
 ** \brief	TMR5_Stop
 **			停止定时器5
 ** \param [in] none
 ** \return   none
 ** \note   									  
 *****************************************************************************/
void TMR5_Stop(void)
{
	T5CON &= ~TMR_T5CON_T5E_Msk;
}
/*****************************************************************************
 ** \brief	TMR5_EnableWakeUp
 **			使能定时器5唤醒
 ** \param [in] none
 ** \return   none
 ** \note   									  
 *****************************************************************************/
void TMR5_EnableWakeUp(void)
{
	T5CON |= TMR_T5CON_I5WUE_Msk;
}
/*****************************************************************************
 ** \brief	TMR5_DisableWakeUp
 **			禁止定时器5唤醒
 ** \param [in] none
 ** \return   none
 ** \note   									  
 *****************************************************************************/
void TMR5_DisableWakeUp(void)
{
	T5CON &= ~TMR_T5CON_I5WUE_Msk;
}
/*****************************************************************************
 ** \brief	TMR5_CLK_Select
 **			选择定时器5时钟源
 ** \param [in] TMR5Sr: TMR5_CLKSR_SYSTEM, TMR5_CLKSR_LSI
 ** \return   none
 ** \note   									  
 *****************************************************************************/
void TMR5_CLK_Select(uint8_t TMR5Sr)
{
	uint8_t Temp=0;
	
	Temp = T5CON;
	Temp &= ~TMR_T5CON_T5CKSn_Msk;
	Temp |= TMR5Sr;
	T5CON = Temp;
}
/*****************************************************************************
 ** \brief	TMR5_FRQ_Set
 **			设置定时器5分频
 ** \param [in] TMR5Frq: TMR5_FRQ_DIV1,TMR5_FRQ_DIV2, ......  TMR5_FRQ_DIV128
 ** \return   none
 ** \note   									  
 *****************************************************************************/
void TMR5_FRQ_Set(uint8_t TMR5Frq)
{
	uint8_t Temp=0;
	
	Temp = T5CON;
	Temp &= ~TMR_T5CON_T5PSCn_Msk;
	Temp |= TMR5Frq;
	T5CON = Temp;	
}
/***************************************************************************
 ** \brief	TMR_ConfigTimerPeriod
 **			配置定时器5定时周期
 ** \param [in] 
 **				TimerPeriodH: TH5 
 **				TimerPeriodL: TL5 
 ** \return  none
 ** \note    16位定时器 
*****************************************************************************/
void TMR5_ConfigPeriod(uint16_t TimerPeriod)
{
	TH5 = (uint8_t)(TimerPeriod>>8);
	TL5 = (uint8_t)TimerPeriod;
}
/***************************************************************************
 ** \brief	TMR5_INT_Enable
 **			使能定时器5中断
 ** \param [in] none
 ** \return  none
 ** \note    
*****************************************************************************/
void TMR5_INT_Enable(void)
{
	EIE2 |= IRQ_EIE2_ET5IE_Msk;
}
/***************************************************************************
 ** \brief	TMR5_INT_Disable
 **			禁止定时器5中断
 ** \param [in] none
 ** \return  none
 ** \note    
*****************************************************************************/
void TMR5_INT_Disable(void)
{
	EIE2 &= ~IRQ_EIE2_ET5IE_Msk;
}
/*****************************************************************************
 ** \brief	TMR5_GetIntFlag
 **			获取定时器5中断标志
 ** \param [in] none
 ** \return   0：无中断， 1：有中断
 ** \note   									  
 *****************************************************************************/
uint8_t TMR5_GetIntFlag(void)
{
	if(EIF2&IRQ_EIF2_TF5_Msk)
		return 1;
	else
		return 0;
}

/*****************************************************************************
 ** \brief	TMR5_GetIntFlag
 **			清除定时器5中断标志
 ** \param [in] none
 ** \return none
 ** \note   									  
 *****************************************************************************/
void TMR5_ClearIntFlag(void)
{
	EIF2 =~IRQ_EIF2_TF5_Msk;
}

